define([
    'underscore',
    'backbone',
    'modules/new-appointment-request/resources/request/model',
    'modules/new-appointment-request/tests/helpers/general-helper',
], function(_, Backbone, Model, helper) {
    'use strict';


    describe('Request Scheduling Model', function() {
        var model;

        beforeEach(function() {
            model = new Model();
            helper.userSpy();
        });

        it('creates the correct url', function() {
            var expected = '../VeteranAppointmentRequestService/v4/rest/appointment-service/patient/ICN/123/appointments';
            expect(model.getUrl()).toBe(expected);
        });

        it('creates the correct request data payload', function() {
            var typeOfCare = new Backbone.Model({
                name: 'Zombie Face Lifts',
                id: 'test',
            });
            var facility = new Model({
                institutionCode: '123AB',
                rootStationCode: '123',
            });
            var data;

            model.set('typeOfCare', typeOfCare);
            model.set('facility', facility);
            data = model.createSaveData();

            expect(data.typeOfCare).toBe(typeOfCare.get('id'));
            expect(data.typeOfCareId).toBe(typeOfCare.get('id'));
            expect(data.requestPhoneCall).toBeFalsy();
            expect(data.providerOption).toBe('');
            expect(data.facility.get('facilityCode')).toBe('123AB');
            expect(data.facility.get('parentSiteCode')).toBe('123');
        });

        it('sets the time options correctly for requests', function() {
            model.set('optionDate1', '01/15/32');
            model.set('optionDate2', '02/16/33');
            model.set('optionTime1', 'PM');

            model.setRequestTimes(model);

            expect(model.get('optionTime2')).toBe('AM');
            expect(model.get('optionDate3')).toBe('No Date Selected');
            expect(model.get('optionTime3')).toBe('No Time Selected');
        });

        describe('save', function() {
            var a;
            var o;

            function clean(attr) {
                return _(attr)
                    .omit('facility')
                    .omit('typeOfCare')
                    .omit('typeOfCareId')
                    .omit('emailPreferences')
                    .values()
                    .value();
            }

            beforeEach(function() {
                spyOn(Backbone.Model.prototype, 'save').and.callFake(function(_attributes, _options) {
                    a = _attributes;
                    o = _options;
                });
            });

            it('saves the model with the correct data', function() {
                var attributes;
                var options;

                model.set('facility', new Backbone.Model({institutionCode: '123AB'}));
                model.set('typeOfCare', new Backbone.Model({id: '1'}));
                model.set('emailPreferences', new Backbone.Model({
                    emailAddress: 'test@gmail.com',
                    emailAllowed: true,
                }));
                // model.set('emailAllowed', true);

                model.save();

                attributes = {
                    facility: {
                        cid: model.get('facility').cid,
                        attributes: {institutionCode: '123AB', facilityCode: '123', parentSiteCode: '123'},
                        _changing: false,
                        _previousAttributes: {institutionCode: '123AB', facilityCode: '123'},
                        changed: {parentSiteCode: '123'},
                        _pending: false,
                    },
                    typeOfCare: '1',
                    emailPreferences: '',
                    optionDate1: 'No Date Selected',
                    optionTime1: 'No Time Selected',
                    optionDate2: 'No Date Selected',
                    optionTime2: 'No Time Selected',
                    optionDate3: 'No Date Selected',
                    optionTime3: 'No Time Selected',
                    typeOfCareId: '1',
                    visitType: 'Office Visit',
                    requestedPhoneCall: false,
                    providerOption: '',
                    email: 'test@gmail.com',
                };

                options = {
                    contentType: 'application/json',
                    dataType: 'json',
                    traditional: true,
                    wait: true,
                    url: '../VeteranAppointmentRequestService/v4/rest/appointment-service/patient/ICN/123/appointments',
                };

                // The facility model was having issues with comparisons, this is work around
                expect(_.keys(a)).toEqual(_.keys(attributes));
                expect(clean(a)).toEqual(clean(attributes));
                expect(a.attributes).toEqual(attributes.attributes);

                expect(o).toEqual(options);
            });
        });
    });
});
